//GEMINI-pro DATASET v1.0 Category: Image Editor ; Style: high level of detail
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

typedef struct {
    int width;
    int height;
    unsigned char *data;
} Image;

Image *load_image(const char *filename) {
    FILE *fp = fopen(filename, "rb");
    if (!fp) {
        fprintf(stderr, "Error opening file: %s\n", filename);
        return NULL;
    }

    // Read the header
    unsigned char header[10];
    if (fread(header, 1, 10, fp) != 10) {
        fprintf(stderr, "Error reading header\n");
        fclose(fp);
        return NULL;
    }

    // Check if the file is a valid BMP file
    if (header[0] != 'B' || header[1] != 'M') {
        fprintf(stderr, "Error: not a valid BMP file\n");
        fclose(fp);
        return NULL;
    }

    // Read the image dimensions
    int offset = *(int *)(header + 10);
    int width = *(int *)(header + 18);
    int height = *(int *)(header + 22);

    // Read the image data
    unsigned char *data = malloc(width * height * 3);
    if (!data) {
        fprintf(stderr, "Error allocating memory\n");
        fclose(fp);
        return NULL;
    }

    if (fseek(fp, offset, SEEK_SET) != 0) {
        fprintf(stderr, "Error seeking to image data\n");
        fclose(fp);
        free(data);
        return NULL;
    }

    if (fread(data, 1, width * height * 3, fp) != width * height * 3) {
        fprintf(stderr, "Error reading image data\n");
        fclose(fp);
        free(data);
        return NULL;
    }

    fclose(fp);

    Image *image = malloc(sizeof(Image));
    if (!image) {
        fprintf(stderr, "Error allocating memory\n");
        free(data);
        return NULL;
    }

    image->width = width;
    image->height = height;
    image->data = data;

    return image;
}

void save_image(const char *filename, Image *image) {
    FILE *fp = fopen(filename, "wb");
    if (!fp) {
        fprintf(stderr, "Error opening file: %s\n", filename);
        return;
    }

    // Write the header
    unsigned char header[10] = {
        'B', 'M', 0, 0, 0, 0, 0, 0, 0, 0
    };

    *(int *)(header + 2) = 54 + image->width * image->height * 3;
    *(int *)(header + 18) = image->width;
    *(int *)(header + 22) = image->height;

    if (fwrite(header, 1, 10, fp) != 10) {
        fprintf(stderr, "Error writing header\n");
        fclose(fp);
        return;
    }

    // Write the image data
    if (fwrite(image->data, 1, image->width * image->height * 3, fp) != image->width * image->height * 3) {
        fprintf(stderr, "Error writing image data\n");
        fclose(fp);
        return;
    }

    fclose(fp);
}

void invert_image(Image *image) {
    for (int i = 0; i < image->width * image->height * 3; i += 3) {
        image->data[i] = 255 - image->data[i];
        image->data[i + 1] = 255 - image->data[i + 1];
        image->data[i + 2] = 255 - image->data[i + 2];
    }
}

int main(int argc, char **argv) {
    if (argc != 3) {
        fprintf(stderr, "Usage: %s <input file> <output file>\n", argv[0]);
        return 1;
    }

    Image *image = load_image(argv[1]);
    if (!image) {
        return 1;
    }

    invert_image(image);

    save_image(argv[2], image);

    free(image->data);
    free(image);

    return 0;
}