//GEMINI-pro DATASET v1.0 Category: Image Editor ; Style: enthusiastic
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

typedef struct
{
    int width;
    int height;
    int max_value;
    unsigned char *data;
} Image;

Image *read_image(char *filename)
{
    FILE *fp = fopen(filename, "rb");
    if (fp == NULL)
    {
        perror("Error opening file");
        return NULL;
    }

    char header[10];
    fread(header, 10, 1, fp);
    if (strcmp(header, "P5") != 0)
    {
        fprintf(stderr, "Error: not a PGM file\n");
        fclose(fp);
        return NULL;
    }

    int width, height, max_value;
    fscanf(fp, "%d %d %d", &width, &height, &max_value);

    Image *image = malloc(sizeof(Image));
    image->width = width;
    image->height = height;
    image->max_value = max_value;
    image->data = malloc(width * height * sizeof(unsigned char));
    fread(image->data, width * height, 1, fp);
    fclose(fp);

    return image;
}

void write_image(Image *image, char *filename)
{
    FILE *fp = fopen(filename, "wb");
    if (fp == NULL)
    {
        perror("Error opening file");
        return;
    }

    fprintf(fp, "P5\n%d %d\n%d\n", image->width, image->height, image->max_value);
    fwrite(image->data, image->width * image->height, 1, fp);
    fclose(fp);
}

void invert(Image *image)
{
    for (int i = 0; i < image->width * image->height; i++)
    {
        image->data[i] = 255 - image->data[i];
    }
}

void blur(Image *image, int radius)
{
    int width = image->width;
    int height = image->height;

    unsigned char *blurred_data = malloc(width * height * sizeof(unsigned char));

    for (int i = 0; i < height; i++)
    {
        for (int j = 0; j < width; j++)
        {
            int sum = 0;
            int count = 0;

            for (int k = -radius; k <= radius; k++)
            {
                for (int l = -radius; l <= radius; l++)
                {
                    int x = j + l;
                    int y = i + k;

                    if (x >= 0 && x < width && y >= 0 && y < height)
                    {
                        sum += image->data[y * width + x];
                        count++;
                    }
                }
            }

            blurred_data[i * width + j] = sum / count;
        }
    }

    free(image->data);
    image->data = blurred_data;
}

void sharpen(Image *image, int radius)
{
    int width = image->width;
    int height = image->height;

    unsigned char *sharpened_data = malloc(width * height * sizeof(unsigned char));

    for (int i = 0; i < height; i++)
    {
        for (int j = 0; j < width; j++)
        {
            int sum = 0;
            int count = 0;

            for (int k = -radius; k <= radius; k++)
            {
                for (int l = -radius; l <= radius; l++)
                {
                    int x = j + l;
                    int y = i + k;

                    if (x >= 0 && x < width && y >= 0 && y < height)
                    {
                        sum += image->data[y * width + x];
                        count++;
                    }
                }
            }

            sharpened_data[i * width + j] = sum / count - image->data[i * width + j];
        }
    }

    free(image->data);
    image->data = sharpened_data;
}

void edge_detection(Image *image, int threshold)
{
    int width = image->width;
    int height = image->height;

    unsigned char *edges_data = malloc(width * height * sizeof(unsigned char));

    for (int i = 0; i < height; i++)
    {
        for (int j = 0; j < width; j++)
        {
            int dx = image->data[(i + 1) * width + j] - image->data[(i - 1) * width + j];
            int dy = image->data[i * width + j + 1] - image->data[i * width + j - 1];

            int gradient = sqrt(dx * dx + dy * dy);

            edges_data[i * width + j] = gradient > threshold ? 255 : 0;
        }
    }

    free(image->data);
    image->data = edges_data;
}

int main(int argc, char **argv)
{
    if (argc < 2)
    {
        printf("Usage: %s <image.pgm>\n", argv[0]);
        return 1;
    }

    Image *image = read_image(argv[1]);
    if (image == NULL)
    {
        return 1;
    }

    char *output_filename = "output.pgm";

    invert(image);
    write_image(image, output_filename);
    printf("Inverted image written to %s\n", output_filename);

    blur(image, 2);
    output_filename = "blurred.pgm";
    write_image(image, output_filename);
    printf("Blurred image written to %s\n", output_filename);

    sharpen(image, 2);
    output_filename = "sharpened.pgm";
    write_image(image, output_filename);
    printf("Sharpened image written to %s\n", output_filename);

    edge_detection(image, 128);
    output_filename = "edges.pgm";
    write_image(image, output_filename);
    printf("Edge-detected image written to %s\n", output_filename);

    free(image->data);
    free(image);

    return 0;
}