//GEMINI-pro DATASET v1.0 Category: Log analysis ; Style: standalone
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

// Define the maximum length of a log line
#define MAX_LOG_LINE_LENGTH 1024

// Define the maximum number of log lines to store in memory
#define MAX_LOG_LINES 1000

// Define the structure of a log line
typedef struct log_line {
    time_t timestamp;
    char *message;
} log_line;

// Define the structure of a log file
typedef struct log_file {
    FILE *file;
    int num_lines;
    log_line *lines;
} log_file;

// Create a new log file
log_file *log_file_new(const char *filename) {
    log_file *lf = malloc(sizeof(log_file));
    if (lf == NULL) {
        return NULL;
    }

    lf->file = fopen(filename, "r");
    if (lf->file == NULL) {
        free(lf);
        return NULL;
    }

    lf->num_lines = 0;
    lf->lines = NULL;

    return lf;
}

// Free a log file
void log_file_free(log_file *lf) {
    if (lf == NULL) {
        return;
    }

    if (lf->file != NULL) {
        fclose(lf->file);
    }

    if (lf->lines != NULL) {
        for (int i = 0; i < lf->num_lines; i++) {
            free(lf->lines[i].message);
        }
        free(lf->lines);
    }

    free(lf);
}

// Read the next log line from a log file
log_line *log_file_read_line(log_file *lf) {
    if (lf == NULL || lf->file == NULL) {
        return NULL;
    }

    char line[MAX_LOG_LINE_LENGTH];
    if (fgets(line, MAX_LOG_LINE_LENGTH, lf->file) == NULL) {
        return NULL;
    }

    // Parse the log line
    log_line *ll = malloc(sizeof(log_line));
    if (ll == NULL) {
        return NULL;
    }

    ll->timestamp = time(NULL);
    ll->message = strdup(line);
    if (ll->message == NULL) {
        free(ll);
        return NULL;
    }

    lf->num_lines++;

    return ll;
}

// Print a log line
void log_line_print(log_line *ll) {
    if (ll == NULL) {
        return;
    }

    printf("%s: %s", ctime(&ll->timestamp), ll->message);
}

// Main function
int main(int argc, char *argv[]) {
    if (argc < 2) {
        printf("Usage: %s <log file>\n", argv[0]);
        return 1;
    }

    // Open the log file
    log_file *lf = log_file_new(argv[1]);
    if (lf == NULL) {
        printf("Error opening log file: %s\n", argv[1]);
        return 1;
    }

    // Read the log lines
    while (1) {
        log_line *ll = log_file_read_line(lf);
        if (ll == NULL) {
            break;
        }

        // Print the log line
        log_line_print(ll);

        // Free the log line
        free(ll);
    }

    // Free the log file
    log_file_free(lf);

    return 0;
}