//GEMINI-pro DATASET v1.0 Category: CPU Scheduling Algorithms ; Style: paranoid
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

// CPU scheduling algorithms
enum algorithms { FCFS, SJF, RR };

// Process structure
struct process {
  int pid;             // Process ID
  int arrival_time;    // Arrival time
  int burst_time;      // Burst time
  int remaining_time;  // Remaining time
  int priority;        // Priority
  int state;           // State (0: running, 1: ready, 2: waiting)
};

// CPU scheduler
struct scheduler {
  int num_processes;     // Number of processes
  struct process *processes;  // Array of processes
  int current_time;      // Current time
  int algorithm;          // Scheduling algorithm
  int quantum;            // Quantum (for RR)
};

// Compare processes by arrival time
int compare_arrival_time(const void *a, const void *b) {
  const struct process *process1 = a;
  const struct process *process2 = b;
  return process1->arrival_time - process2->arrival_time;
}

// Compare processes by burst time
int compare_burst_time(const void *a, const void *b) {
  const struct process *process1 = a;
  const struct process *process2 = b;
  return process1->burst_time - process2->burst_time;
}

// Compare processes by priority
int compare_priority(const void *a, const void *b) {
  const struct process *process1 = a;
  const struct process *process2 = b;
  return process1->priority - process2->priority;
}

// Initialize scheduler
void init_scheduler(struct scheduler *scheduler, int num_processes, struct process *processes, int algorithm, int quantum) {
  scheduler->num_processes = num_processes;
  scheduler->processes = processes;
  scheduler->current_time = 0;
  scheduler->algorithm = algorithm;
  scheduler->quantum = quantum;
}

// Run scheduler
void run_scheduler(struct scheduler *scheduler) {
  int i;
  struct process *running_process = NULL;

  // Sort processes by arrival time
  qsort(scheduler->processes, scheduler->num_processes, sizeof(struct process), compare_arrival_time);

  // While there are processes to run
  while (1) {
    // Find the next process to run
    for (i = 0; i < scheduler->num_processes; i++) {
      if (scheduler->processes[i].state == 1) {  // Ready
        if (running_process == NULL || running_process->state != 0) {  // Running
          running_process = &scheduler->processes[i];
          break;
        }
      }
    }

    // If there are no more processes to run, break
    if (running_process == NULL) {
      break;
    }

    // Switch to the new process
    running_process->state = 0;  // Running

    // Update current time
    scheduler->current_time += running_process->remaining_time;

    // Update remaining time
    running_process->remaining_time = 0;

    // Check the scheduling algorithm
    switch (scheduler->algorithm) {
      case FCFS:
        // Do nothing
        break;
      case SJF:
        // Sort processes by burst time
        qsort(scheduler->processes, scheduler->num_processes, sizeof(struct process), compare_burst_time);
        break;
      case RR:
        // Decrement remaining time
        running_process->remaining_time -= scheduler->quantum;
        if (running_process->remaining_time > 0) {
          running_process->state = 1;  // Ready
        } else {
          running_process->state = 2;  // Waiting
        }
        break;
    }

    // If the process is finished, remove it from the list
    if (running_process->remaining_time == 0) {
      for (i = 0; i < scheduler->num_processes; i++) {
        if (scheduler->processes[i].pid == running_process->pid) {
          scheduler->processes[i] = scheduler->processes[scheduler->num_processes - 1];
          scheduler->num_processes--;
          break;
        }
      }
    }

    // Switch to the next process
    running_process = NULL;
  }
}

// Print scheduler results
void print_scheduler_results(struct scheduler *scheduler) {
  int i;
  double avg_waiting_time = 0;
  double avg_turnaround_time = 0;

  // Calculate average waiting time and turnaround time
  for (i = 0; i < scheduler->num_processes; i++) {
    avg_waiting_time += scheduler->processes[i].burst_time;
    avg_turnaround_time += scheduler->processes[i].burst_time + scheduler->processes[i].remaining_time;
  }
  avg_waiting_time /= scheduler->num_processes;
  avg_turnaround_time /= scheduler->num_processes;

  // Print results
  printf("Scheduling algorithm: %s\n", scheduler->algorithm == FCFS ? "FCFS" : scheduler->algorithm == SJF ? "SJF" : "RR");
  printf("Average waiting time: %.2f\n", avg_waiting_time);
  printf("Average turnaround time: %.2f\n", avg_turnaround_time);
}

// Main function
int main() {
  // Initialize processes
  struct process processes[] = {
    {1, 0, 5, 5, 1, 1},
    {2, 1, 3, 3, 2, 1},
    {3, 2, 4, 4, 3, 1},
    {4, 3, 2, 2, 4, 1},
    {5, 4, 1, 1, 5, 1},
  };

  // Initialize scheduler
  struct scheduler scheduler;
  init_scheduler(&scheduler, 5, processes, FCFS, 0);

  // Run scheduler
  run_scheduler(&scheduler);

  // Print scheduler results
  print_scheduler_results(&scheduler);

  return 0;
}