//GPT-4o-mini DATASET v1.0 Category: Building a JSON Parser ; Style: recursive
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

typedef struct JsonValue {
    enum { JSON_NULL, JSON_TRUE, JSON_FALSE, JSON_NUMBER, JSON_STRING, JSON_ARRAY, JSON_OBJECT } type;
    union {
        double number;
        char* string;
        struct JsonValue* array; // pointer to first element of an array
        struct JsonKeyValuePair* object; // pointer to first key-value pair
    } value;
} JsonValue;

typedef struct JsonKeyValuePair {
    char* key;
    JsonValue value;
    struct JsonKeyValuePair* next;
} JsonKeyValuePair;

void freeJsonValue(JsonValue* value);
void freeJsonKeyValuePair(JsonKeyValuePair* pair);
void parseWhitespace(const char** json);
JsonValue parseValue(const char** json);

JsonValue parseJson(const char* json) {
    parseWhitespace(&json);
    return parseValue(&json);
}

void parseWhitespace(const char** json) {
    while (isspace(**json)) (*json)++;
}

JsonValue parseValue(const char** json) {
    JsonValue value;
    switch (**json) {
        case 'n': {
            if (strncmp(*json, "null", 4) == 0) {
                value.type = JSON_NULL;
                *json += 4;
                return value;
            }
            break;
        }
        case 't': {
            if (strncmp(*json, "true", 4) == 0) {
                value.type = JSON_TRUE;
                *json += 4;
                return value;
            }
            break;
        }
        case 'f': {
            if (strncmp(*json, "false", 5) == 0) {
                value.type = JSON_FALSE;
                *json += 5;
                return value;
            }
            break;
        }
        case '"': {
            // Handle string parsing
            (*json)++;
            const char* start = *json;
            while (**json && **json != '"') {
                if (**json == '\\') (*json)++; // skip escape character
                (*json)++;
            }
            size_t length = *json - start;
            value.type = JSON_STRING;
            value.value.string = malloc(length + 1);
            strncpy(value.value.string, start, length);
            value.value.string[length] = '\0';
            (*json)++;
            return value;
        }
        case '[': {
            // Handle array parsing
            value.type = JSON_ARRAY;
            (*json)++;
            // Current logic will be expanded later
            return value;
        }
        case '{': {
            // Handle object parsing
            value.type = JSON_OBJECT;
            (*json)++;
            // Current logic will be expanded later
            return value;
        }
        default: {
            if (isdigit(**json) || **json == '-') {
                // Handle number parsing
                char* end;
                value.type = JSON_NUMBER;
                value.value.number = strtod(*json, &end);
                *json = end;
                return value;
            }
        }
    }
    fprintf(stderr, "Error parsing JSON: unexpected character: '%c'\n", **json);
    exit(EXIT_FAILURE);
}

// Print JSON value in an indented format
void printJsonValue(JsonValue value, int indent) {
    switch (value.type) {
        case JSON_NULL:
            printf("null");
            break;
        case JSON_TRUE:
            printf("true");
            break;
        case JSON_FALSE:
            printf("false");
            break;
        case JSON_NUMBER:
            printf("%f", value.value.number);
            break;
        case JSON_STRING:
            printf("\"%s\"", value.value.string);
            break;
        case JSON_ARRAY:
            // Arrays are not implemented in this example
            break;
        case JSON_OBJECT:
            // Objects are not implemented in this example
            break;
    }
}

// Free allocated memory for JsonValue
void freeJsonValue(JsonValue* value) {
    if (value->type == JSON_STRING) {
        free(value->value.string);
    }
    // Add logic to free arrays and objects when implemented
}

// Sample main function
int main() {
    const char* json_input = "{\"name\": \"John\", \"age\": 30, \"isEmployed\": true, \"address\": null}";
    JsonValue parsed = parseJson(json_input);
    printJsonValue(parsed, 0);
    freeJsonValue(&parsed);
    return 0;
}