//GPT-4o-mini DATASET v1.0 Category: Classical Circuit Simulator ; Style: paranoid
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#define MAX_COMPONENTS 10
#define MAX_NODES 5

typedef struct {
    char type[10]; // "resistor", "capacitor", "inductor", "voltage_source", "current_source"
    double value;  // The value of the component (ohms, farads, henrys, volts, amps)
    int node1;     // First node connection
    int node2;     // Second node connection
} Component;

typedef struct {
    double voltages[MAX_NODES];
    double currents[MAX_NODES];
} CircuitState;

Component circuit[MAX_COMPONENTS];
CircuitState state;

void initializeCircuit() {
    for (int i = 0; i < MAX_NODES; i++) {
        state.voltages[i] = 0;
        state.currents[i] = 0;
    }
}

void addComponent(int index, const char *type, double value, int node1, int node2) {
    if (index < 0 || index >= MAX_COMPONENTS) {
        fprintf(stderr, "Attempting to add a component out of bounds! Suspicious...\n");
        exit(EXIT_FAILURE);
    }
    
    strcpy(circuit[index].type, type);
    circuit[index].value = value;
    circuit[index].node1 = node1;
    circuit[index].node2 = node2;
    printf("Added %s: %f between nodes %d and %d.\n", type, value, node1, node2);
}

double calculateEquivalentResistance() {
    double req = 0;
    for (int i = 0; i < MAX_COMPONENTS && strlen(circuit[i].type) > 0; i++) {
        if (strcmp(circuit[i].type, "resistor") == 0) {
            if (circuit[i].node1 == 0 && circuit[i].node2 == 1) {
                req += circuit[i].value;
                printf("Calculating resistance from component[%d]: Resistance = %f.\n", i, circuit[i].value);
            }
        }
    }
    return req;
}

double calculateTotalCurrent() {
    double totalCurrent = 0;
    for (int i = 0; i < MAX_COMPONENTS && strlen(circuit[i].type) > 0; i++) {
        if (strcmp(circuit[i].type, "current_source") == 0) {
            totalCurrent += circuit[i].value;
            printf("Summing current from component[%d]: Current = %f.\n", i, circuit[i].value);
        }
    }
    return totalCurrent;
}

void simulate() {
    printf("Starting to simulate the circuit...\n");
    
    double req = calculateEquivalentResistance();
    double totalCurrent = calculateTotalCurrent();
    
    for (int i = 0; i < MAX_NODES; i++) {
        if (i == 0) {
            state.voltages[i] = totalCurrent * req;
        } else {
            state.voltages[i] = state.voltages[i - 1] - (totalCurrent * req);
        }
        printf("Node %d Voltage: %f V\n", i, state.voltages[i]);
    }
    
    printf("Simulation complete.\n");
}

int main() {
    printf("***** PARANOID CIRCUIT SIMULATOR *****\n");
    
    initializeCircuit();
    
    addComponent(0, "resistor", 1000, 0, 1); // 1 kΩ resistor between node 0 and 1
    addComponent(1, "current_source", 5, 1, 0); // 5 A current source
    addComponent(2, "resistor", 2000, 1, 2); // 2 kΩ resistor between node 1 and 2
    addComponent(3, "capacitor", 0.001, 2, 3); // 1 mF capacitor
    
    simulate();
    
    printf("Final Voltages:\n");
    for (int i = 0; i < MAX_NODES; i++) {
        printf("Node %d Voltage: %f V\n", i, state.voltages[i]);
    }

    return 0;
}