//GEMINI-pro DATASET v1.0 Category: Password management ; Style: configurable
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <openssl/sha.h>

#define MAX_PASSWORDS 100
#define MAX_PASSWORD_LENGTH 128

typedef struct {
    char username[MAX_PASSWORD_LENGTH];
    char password[MAX_PASSWORD_LENGTH];
    char salt[16];
} password_entry;

// Global array of password entries
password_entry passwords[MAX_PASSWORDS];

// Number of password entries
int num_passwords = 0;

// Configuration options
bool log_in_console = false;
bool save_to_file = false;
char *config_file = "passwords.conf";

// Function to hash a password
void hash_password(char *password, char *hashed_password) {
    SHA256_CTX sha256;
    SHA256_Init(&sha256);
    SHA256_Update(&sha256, password, strlen(password));
    SHA256_Final((unsigned char *)hashed_password, &sha256);
}

// Function to add a new password entry
void add_password(char *username, char *password, char *salt) {
    // Check if the array is full
    if (num_passwords >= MAX_PASSWORDS) {
        printf("Error: Password array is full\n");
        return;
    }

    strcpy(passwords[num_passwords].username, username);

    // Hash the password
    hash_password(password, passwords[num_passwords].password);

    // Copy the salt
    strcpy(passwords[num_passwords].salt, salt);

    num_passwords++;
}

// Function to find a password entry by username
password_entry *find_password(char *username) {
    for (int i = 0; i < num_passwords; i++) {
        if (strcmp(passwords[i].username, username) == 0) {
            return &passwords[i];
        }
    }

    return NULL;
}

// Function to verify a password
bool verify_password(char *username, char *password) {
    password_entry *entry = find_password(username);

    if (entry == NULL) {
        return false;
    }

    // Hash the password
    char hashed_password[SHA256_DIGEST_LENGTH];
    hash_password(password, hashed_password);

    // Compare the hashed passwords
    return memcmp(hashed_password, entry->password, SHA256_DIGEST_LENGTH) == 0;
}

// Function to load passwords from a file
void load_passwords(char *filename) {
    FILE *fp = fopen(filename, "r");
    if (fp == NULL) {
        printf("Error: Could not open file %s\n", filename);
        return;
    }

    char line[1024];
    while (fgets(line, sizeof(line), fp) != NULL) {
        // Parse the line
        char *username = strtok(line, ":");
        char *password = strtok(NULL, ":");
        char *salt = strtok(NULL, ":");

        // Add the password entry
        add_password(username, password, salt);
    }

    fclose(fp);
}

// Function to save passwords to a file
void save_passwords(char *filename) {
    FILE *fp = fopen(filename, "w");
    if (fp == NULL) {
        printf("Error: Could not open file %s\n", filename);
        return;
    }

    for (int i = 0; i < num_passwords; i++) {
        fprintf(fp, "%s:%s:%s\n", passwords[i].username, passwords[i].password, passwords[i].salt);
    }

    fclose(fp);
}

// Function to print the password entries
void print_passwords() {
    for (int i = 0; i < num_passwords; i++) {
        printf("Username: %s\n", passwords[i].username);
        printf("Password: %s\n", passwords[i].password);
        printf("Salt: %s\n", passwords[i].salt);
        printf("\n");
    }
}

int main() {
    // Load passwords from file
    if (save_to_file) {
        load_passwords(config_file);
    }

    // Add a new password entry
    add_password("username1", "password1", "salt1");

    // Verify a password
    bool verified = verify_password("username1", "password1");
    if (verified) {
        printf("Password verified\n");
    } else {
        printf("Password not verified\n");
    }

    // Print the password entries
    if (log_in_console) {
        print_passwords();
    }

    // Save passwords to file
    if (save_to_file) {
        save_passwords(config_file);
    }

    return 0;
}